<?php
/* NOTICE OF LICENSE
 *
 * This source file is subject to a commercial license from SARL SMC
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the SARL SMC is strictly forbidden.
 * In order to obtain a license, please contact us: olivier@common-services.com
 * ...........................................................................
 * INFORMATION SUR LA LICENCE D'UTILISATION
 *
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe SMC
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part de la SARL SMC est
 * expressement interdite.
 * Pour obtenir une licence, veuillez contacter la SARL SMC a l'adresse: olivier@common-services.com
 * ...........................................................................
 * @package    Amazon Market Place
 * @copyright  Copyright (c) 2011-2013 S.A.R.L SMC (http://www.common-services.com)
 * @copyright  Copyright (c) 2011-2013 Common Services Co Ltd - 90/25 Sukhumvit 81 - 10260 Bangkok - Thailand
 * @copyright  Copyright (c) 2011-2013 Olivier B.
 * @author     Olivier B.
 * @license    Commercial license
 * Support by mail  :  contact@common-services.com
 * Support on forum :  delete
 * Skype : delete13_fr
 * Phone : +33.970465505
 */

require_once(dirname(__FILE__) . '/../classes/amazon.order.class.php');
require_once(dirname(__FILE__) . '/../classes/amazon.tools.class.php');

class Amazon_Messaging extends Amazon
{
    private $_debug;

    static $invoice_subjects =
            array(
               'fr' => 'Facture pour votre commande',
               'en' => 'Invoice for your order',
               'de' => 'Rechnung f&uuml;r ihre bestellung',
               'it' => 'Fattura per il vostro ordine',
               'es' => 'Factura de su pedido'
            ) ;
    
    static $review_subjects =
            array(
               'fr' => 'Evaluation concernant votre commande N&ordm;',
               'en' => 'Seller Rating for your order No.',
               'de' => 'Bewertung Ihrer Bestellung Nr.',
               'it' => 'Valutazione del Suo ordine N&ordm;',
               'es' => 'Evaluaci&oacute;n relativa a su pedido N&ordm;'
            ) ;    
    public function __construct($debug = false)
    {
        if ($debug)
            $this->_debug = true;
        else
            $this->_debug = false;

        $this->path = _PS_MODULE_DIR_ . $this->name . '/';
        $this->path_pdf = $this->path . 'pdf/';
        $this->path_mail = $this->path . 'mail/';
        
        parent::__construct() ;
    }

    public function sendInvoice($id_order)
    {
        $mail_invoice = unserialize(base64_decode(Amazon::configurationGet('AMAZON_MAIL_INVOICE')));
        $account_type = unserialize(base64_decode(Amazon::configurationGet('AMAZON_ACCOUNT_TYPE')));

        if (!isset($mail_invoice['active']) || !(int) $mail_invoice['active'])
        {
            return(false);
        }

        if (!isset($mail_invoice['template']) || empty($mail_invoice['template']))
        {
            if ($this->_debug)
                printf('%s:#%d You must select an email template' . "<br />\n", basename(__FILE__), __LINE__);
            return(false);
        }

        if ($account_type != Amazon::AccounTypeIndividual)
        {
            if ($this->_debug)
                printf('%s:#%d This feature can work only with global account configuration' . "<br />\n", basename(__FILE__), __LINE__);
            return(false);
        }

        $order = new Amazon_Order($id_order);

        if (!Validate::isLoadedObject($order) || ! $order->id_lang || !isset($order->marketPlaceOrderId) || empty($order->marketPlaceOrderId) )
        {
            if ($this->_debug)
                printf('%s:#%d Invalid Order (%d)' . "<br />\n", basename(__FILE__), __LINE__, $id_order);
            return(false);
        }

        if (!$order->invoice_number)
        {
            if ($this->_debug)
                printf('%s:#%d Invalid processing for Order (%d) - Order has no invoice number' . "<br />\n", basename(__FILE__), __LINE__, $id_order);
            return(false);
        }

        $customer = new Customer($order->id_customer);

        if (!Validate::isLoadedObject($customer))
        {
            if ($this->_debug)
                printf('%s:#%d Invalid Customer (%d)' . "<br />\n", basename(__FILE__), __LINE__, $order->id_customer);
            return(false);
        }

        $id_lang = $order->id_lang;
        $lang = Language::getIsoById($id_lang);

        $template_file = sprintf('%s%s/%s.html', $this->path_mail, $lang, $mail_invoice['template']) ;

        if ( ! file_exists($template_file) )
        {
            if ($this->_debug)
                printf('%s:#%d Template file doesn\'t exists for this lang: %s(%d)' . "<br />\n", basename(__FILE__), __LINE__, $lang, $id_lang);
            return(false);
        }        

        $template_vars = array();

        $template_vars['{firstname}'] = htmlentities($customer->firstname, ENT_COMPAT, "UTF-8");
        $template_vars['{lastname}'] = htmlentities($customer->lastname, ENT_COMPAT, "UTF-8");

        $template_vars['{amazon_order_id}'] = $order->marketPlaceOrderId;
        $template_vars['{amazon_order_date}'] = Amazon_Tools::displayDate($order->date_add, $id_lang);

        if  ( isset(self::$invoice_subjects[$lang]) )
             $title = self::$invoice_subjects[$lang] ;
        else $title = self::$invoice_subjects['en'] ;
        
        $email_subject = sprintf('%s %s', html_entity_decode($title, ENT_COMPAT, 'UTF-8'), $order->marketPlaceOrderId);

        $email = $customer->email;
        $to_name = sprintf('%s %s', $customer->firstname, $customer->lastname);

        ob_start(); // prevent output
        
        $file_attachement = array();
        
        if ( version_compare(_PS_VERSION_, '1.5', '>=') )
        {
            $pdf = new PDF($order->getInvoicesCollection(), PDF::TEMPLATE_INVOICE, Context::getContext()->smarty);
            $file_attachement[0]['content'] = $pdf->render(false);
            $file_attachement[0]['name'] = Configuration::get('PS_INVOICE_PREFIX', (int) $order->id_lang, null, $order->id_shop) . sprintf('%06d', $order->invoice_number) . '.pdf';
            $file_attachement[0]['mime'] = 'application/pdf';
            
            if (isset($mail_invoice['additionnal']) && file_exists($this->path_pdf . $mail_invoice['additionnal']))
            {
                $file_attachement[1]['content'] = file_get_contents($this->path_pdf . $mail_invoice['additionnal']);
                $file_attachement[1]['name'] = $mail_invoice['additionnal'];
                $file_attachement[1]['mime'] = 'application/pdf';
            }            
        }
        else
        {
            global $cookie ;
            
            $id_employee = Amazon::configurationGet('AMAZON_EMPLOYEE') ;
            
            if ( ! $cookie )
            {
                $cookie = new Cookie() ;
                $cookie->id_employee = $id_employee ? $id_employee : 1 ;
            }
            else
            {
                $cookie->id_employee = $id_employee ;
            }
            $file_attachement['content'] = PDF::invoice($order, 'S');
            $file_attachement['name'] = Configuration::get('PS_INVOICE_PREFIX', (int) $order->id_lang) . sprintf('%06d', $order->invoice_number) . '.pdf';
            $file_attachement['mime'] = 'application/pdf';
        }
        ob_get_clean(); // prevent output - end

        $result = Mail::Send(
            $id_lang, // id_lang
            $mail_invoice['template'], // template
            $email_subject, // subject
            $template_vars, // templateVars 
            $email, // to
            $to_name, // To Name
            null, // From
            null, // From Name 
            $file_attachement, // Attachment
            null, // SMTP
            $this->path_mail);

        return($result);
    }
    
    public function sendReviewIncentive($id_order)
    {
        $mail_review = unserialize(base64_decode(Amazon::configurationGet('AMAZON_MAIL_REVIEW')));
        $account_type = unserialize(base64_decode(Amazon::configurationGet('AMAZON_ACCOUNT_TYPE')));

        if (!isset($mail_review['active']) || !(int) $mail_review['active'])
        {
            return(false);
        }

        if (!isset($mail_review['template']) || empty($mail_review['template']))
        {
            if ($this->_debug)
                printf('%s:#%d You must select an email template' . "<br />\n", basename(__FILE__), __LINE__);
            return(false);
        }

        if ($account_type != Amazon::AccounTypeIndividual)
        {
            if ($this->_debug)
                printf('%s:#%d This feature can work only with global account configuration' . "<br />\n", basename(__FILE__), __LINE__);
            return(false);
        }

        $order = new Amazon_Order($id_order);

        if (!Validate::isLoadedObject($order) || ! $order->id_lang || !isset($order->marketPlaceOrderId) || empty($order->marketPlaceOrderId) )
        {
            if ($this->_debug)
                printf('%s:#%d Invalid Order (%d)' . "<br />\n", basename(__FILE__), __LINE__, $id_order);
            return(false);
        }

        if (!$order->invoice_number)
        {
            if ($this->_debug)
                printf('%s:#%d Invalid processing for Order (%d) - Order has no invoice number' . "<br />\n", basename(__FILE__), __LINE__, $id_order);
            return(false);
        }

        $dateOrdered = date('Y-m-d', strtotime($order->date_add));
        $dateCurrent = date('Y-m-d');

        if ( (int)$mail_review['delay'] && Amazon_Tools::getWorkingDays($dateOrdered, $dateCurrent) >= (int)$mail_review['delay'] )
        {
            if ($this->_debug)
                printf('%s:#%d Out of delay: created on %s - sent on %s' . "<br />\n", basename(__FILE__), __LINE__, $dateOrdered, $dateCurrent);
            return(false);            
        }
        
        $customer = new Customer($order->id_customer);

        if (!Validate::isLoadedObject($customer))
        {
            if ($this->_debug)
                printf('%s:#%d Invalid Customer (%d)' . "<br />\n", basename(__FILE__), __LINE__, $order->id_customer);
            return(false);
        }

        $id_lang = $order->id_lang;
        $lang = Language::getIsoById($id_lang);
        
        $template_file = sprintf('%s%s/%s.html', $this->path_mail, $lang, $mail_review['template']) ;

        if ( ! file_exists($template_file) )
        {
            if ($this->_debug)
                printf('%s:#%d Template file doesn\'t exists for this lang: %s(%d)' . "<br />\n", basename(__FILE__), __LINE__, $lang, $id_lang);
            return(false);
        }        

        if  ( isset(self::$review_subjects[$lang]) )
             $title = self::$review_subjects[$lang] ;
        else $title = self::$review_subjects['en'] ;
     
        $email_subject = sprintf('%s %s', html_entity_decode($title, ENT_COMPAT, 'UTF-8'), $order->marketPlaceOrderId);
        $email = $customer->email;
        $to_name = sprintf('%s %s', $customer->firstname, $customer->lastname);
        
        $template_vars = array();

        $template_vars['{firstname}'] = htmlentities($customer->firstname, ENT_COMPAT, "UTF-8");
        $template_vars['{lastname}'] = htmlentities($customer->lastname, ENT_COMPAT, "UTF-8");

        $template_vars['{amazon_order_id}'] = $order->marketPlaceOrderId;
        $template_vars['{amazon_order_date}'] = Amazon_Tools::displayDate($order->date_add, $id_lang);
        $template_vars['{amazon_review_url}'] = Amazon_Tools::goToSellerReviewPage($id_lang) ;
        $template_vars['{amazon_review_url_html}'] = sprintf('<a href="%s" title="%s">%s</a>', Amazon_Tools::goToSellerReviewPage($id_lang), $email_subject, Amazon_Tools::goToSellerReviewPage($id_lang)) ;

        $result = Mail::Send(
            $id_lang, // id_lang
            $mail_review['template'], // template
            $email_subject, // subject
            $template_vars, // templateVars 
            $email, // to
            $to_name, // To Name
            null, // From
            null, // From Name 
            null, // Attachment
            null, // SMTP
            $this->path_mail);

        return($result);
    }
    
    
}
