<?php

/**

 * 2007-2018 KU

 *

 * DISCLAIMER

 *

 * Do not edit or add to this file if you wish to upgrade this module to newer

 * versions in the future.

 *

 *  @author    Kevin UNFRICHT <unfricht.kevin@hotmail.fr>

 *  @copyright 2007-2018 Kevin UNFRICHT

 *  @license   Commercial license see license.txt

 *  Property of Kevin UNFRICHT - AZCODE

 */



if (!class_exists('AzcodeAutoload')) {

    class AzcodeAutoload

    {

        /**

        * File where classes index is stored

        */

        const INDEX_FILE = 'class_index.php';



        /**

        * @var AzcodeAutoload

        */

        protected static $instance;



        /**

        * @var string Root directory

        */

        protected $root_dir;



        public $index = array();



        protected function __construct($module_name)

        {

            $this->root_dir = _PS_MODULE_DIR_.$module_name.DIRECTORY_SEPARATOR;

            $file = $this->normalizeDirectory($this->root_dir).AzcodeAutoload::INDEX_FILE;

            if (@filemtime($file) && is_readable($file)) {

                $this->index = include($file);

            } else {

                $this->generateIndex();

            }

        }



        /**

        * Get instance of autoload (singleton)

        *

        * @return AzcodeAutoload

        */

        public static function getInstance($module_name)

        {

            if (!AzcodeAutoload::$instance) {

                AzcodeAutoload::$instance = new AzcodeAutoload($module_name);

            }

            return AzcodeAutoload::$instance;

        }



        /**

        * Retrieve informations about a class in classes index and load it

        *

        * @param string $classname

        */

        public function load($classname)

        {

            // regenerate the class index if the requested file doesn't exists

            if ((isset($this->index[$classname])

                && $this->index[$classname]['path']

                && !is_file($this->root_dir.$this->index[$classname]['path']))) {

                $this->generateIndex();

            }



            // Call directly ProductCore, ShopCore class

            if (isset($this->index[$classname]['path']) && $this->index[$classname]['path']) {

                require_once($this->root_dir.$this->index[$classname]['path']);

            }

        }



        /**

        * Generate classes index

        */

        public function generateIndex()

        {

            $classes = array_merge(

                $this->getClassesFromDir('classes/'),

                $this->getClassesFromDir('vendors/')

            );



            ksort($classes);

            $content = '<?php return '.var_export($classes, true).'; ?>';



            // Write classes index on disc to cache it

            $filename = $this->normalizeDirectory($this->root_dir).AzcodeAutoload::INDEX_FILE;

            $filename_tmp = tempnam(dirname($filename), basename($filename.'.'));

            if ($filename_tmp !== false && file_put_contents($filename_tmp, $content) !== false) {

                if (!@rename($filename_tmp, $filename)) {

                    unlink($filename_tmp);

                } else {

                    @chmod($filename, 0666);

                }

            } else {

                Tools::error_log('Cannot write temporary file '.$filename_tmp);

            }

            $this->index = $classes;

        }



        /**

        * Retrieve recursively all classes in a directory and its subdirectories

        *

        * @param string $path Relativ path from root to the directory

        * @return array

        */

        protected function getClassesFromDir($path)

        {

            $classes = array();

            $root_dir = $this->root_dir;



            foreach (scandir($root_dir.$path) as $file) {

                if ($file[0] != '.') {

                    if (is_dir($root_dir.$path.$file)) {

                        $classes = array_merge($classes, $this->getClassesFromDir($path.$file.'/'));

                    } elseif (Tools::substr($file, -4) == '.php') {

                        $content = Tools::file_get_contents($root_dir.$path.$file);



                        $namespacePattern = '[\\a-z0-9_]*[\\]';

                        $pattern = '#\W((abstract\s+)?class|interface)\s+(?P<classname>'

                            .basename($file, '.php').')'

                            .'(?:\s+extends\s+'.$namespacePattern

                            .'[a-z][a-z0-9_]*)?(?:\s+implements\s+'

                            .$namespacePattern.'[a-z][\\a-z0-9_]*(?:\s*,\s*'

                            .$namespacePattern.'[a-z][\\a-z0-9_]*)*)?\s*\{#i';



                        if (preg_match($pattern, $content, $m)) {

                            $classes[$m['classname']] = array(

                                'path' => $path.$file,

                                'type' => trim($m[1])

                            );

                        }

                    }

                }

            }



            return $classes;

        }



        private function normalizeDirectory($directory)

        {

            return rtrim($directory, '/\\').DIRECTORY_SEPARATOR;

        }

    }

}

