<?php
/*
*  2013 Neovis : Michael Jegou & Vasseur Alexandre
*
*  @author VASSEUR ALEXANDRE <tech@neovis.fr>
*  @author MJ <tech@neovis.fr>
*  @copyright  2013 Neovis
*  @license    not free
*  International Registered Trademark & Property of Neovis
*/

/* for old 1.4 */
if (!defined('_CAN_LOAD_FILES_') && !defined('_PS_ROOT_DIR_'))
	exit;

/* On inclus la lib de traitement de la TVA selon la version de prestashop */
if (version_compare(_PS_VERSION_, '1.5.0.0') >= 0)
	include_once(dirname(__FILE__).'/classes/TVAUpdater15.php');
elseif (version_compare(_PS_VERSION_, '1.4.0.0') >= 0)
	include_once(_PS_ROOT_DIR_.'/modules/neotaxesupdater/classes/TVAUpdater14.php');
elseif (version_compare(_PS_VERSION_, '1.3.0.0') >= 0)
	include_once(_PS_ROOT_DIR_.'/modules/neotaxesupdater/classes/TVAUpdater13.php');

class NeoTaxesUpdater extends Module
{
	/* true => on doit indiquer les nouveaux taux automatiquement, sinon on fait rien */
	public $neotaxesupdater_getdefaultrules = false;

	public function __construct()
	{
		$this->name = 'neotaxesupdater';
		$this->tab = ($this->isVer14AndMore()) ? 'administration' : 'administration'; // @TODO: à verifier pour 1.3
		$this->version = '1.2';
		$this->author = 'Neovis';
		$this->module_key = 'f455dbe2657db73513e03bbb186322a4';

		parent::__construct();

		$this->displayName = $this->l('Neovis Taxes Updater');
		$this->description = $this->l('Easily change the new VAT for your products with this module.');

		$this->confirmUninstall = $this->l('Are you sure you want to uninstall?');

	//	Backward compatibility
		require(_PS_MODULE_DIR_.$this->name.'/backward_compatibility/backward.php');

	//	initialisation de la base ou whatever.
		$this->initialize();
	}

/********************************************************************************************
 *	Si y'a des choses
 *	à créer dans la base de données ou
 *	à enregistrer en base ou quoique ce soit à l'installation le faire ici
 ********************************************************************************************/
	protected function initialize()
	{
	}

/********************************************************************************************
 *	fonction d'installation du module
 *	appel des hook si on en met
 ********************************************************************************************/
	public function install()
	{
	//	on l'installe sur tout les shop
		if ($this->isVer15AndMore() && Shop::isFeatureActive())
			Shop::setContext(Shop::CONTEXT_ALL);

		if (!parent::install()
			|| !$this->registerHook('displayBackOfficeHeader')	// pour la CSS admin
			|| !$this->registerHook('displayHeader') // pour le mode CRON auto
			|| !$this->getCronToken() // generate a token for cron access if it doesn't exists yet
			|| !TVAUpdater::updateDB())
			return false;
		return true;
	}

/********************************************************************************************
 *	génère un token si inexistant et le renvoi, sinon renvoi le token existant en base
 ********************************************************************************************/
	public function getCronToken()
	{
		$token = Configuration::get('NEOTAXESUPDATER_CRONTOKEN');
		if ($token == '')
		{
			$token = TVAUpdater::getAdminTokenLite('Neo'.mt_rand().'Tax'.mt_rand().'Updater');
			TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_CRONTOKEN', $token);
			$token = Configuration::get('NEOTAXESUPDATER_CRONTOKEN'); // Verify that all is saved!
		}
		return $token;
	}
/********************************************************************************************
 *	teste si on est dans le cadre d'une 1.5 ou plus
 ********************************************************************************************/
	public function isVer15AndMore()
	{
		return (version_compare(_PS_VERSION_, '1.5.0.0') >= 0);
	}

/********************************************************************************************
 *	teste si on est dans le cadre d'une 1.4 ou plus
 ********************************************************************************************/
	public function isVer14AndMore()
	{
		return (version_compare(_PS_VERSION_, '1.4.0.0') >= 0);
	}

/********************************************************************************************
 *	teste si on est dans le cadre d'une 1.3 ou plus
 ********************************************************************************************/
	public function isVer13AndMore()
	{
		return (version_compare(_PS_VERSION_, '1.3.0.0') >= 0);
	}

/********************************************************************************************
 *	on override la fonction pour gérer le cas 1.3 et 1.4
 ********************************************************************************************/

	public function registerHook($hookname, $shop_list = null)
	{
		$shop_list = $shop_list; // just for validator
		if ($this->isVer15AndMore())
			return parent::registerHook($hookname, null);
		elseif ($this->isVer13AndMore())
		{
			switch ($hookname)
			{
				case 'displayHeader':
					return parent::registerHook('header');

				case 'displayBackOfficeHeader':
					return true; // géré dans la classe dédiée de compatibilité

				default:
					return parent::registerHook($hookname);
			}
		}
	}

	public function adminDisplayWarning($msg)
	{
		if ($this->isVer15AndMore())
			return parent::adminDisplayWarning($msg);
		else
			echo '<div class="warn"><img src="../img/admin/warn2.png">'.$msg.'</div>'; // @TODO: do it better
	}

	public function adminDisplayInformation($msg)
	{
		if ($this->isVer15AndMore())
			return parent::adminDisplayInformation($msg);
		else
			echo '<div class="alert"><img src="../img/admin/information.png">'.$msg.'</div>'; // @TODO: do it better
	}

/********************************************************************************************
 *	Ajout feuille de style backoffice
 ********************************************************************************************/
	public function hookDisplayBackOfficeHeader()
	{
		if (method_exists(Context::getContext()->controller, 'addCss'))
			Context::getContext()->controller->addCss($this->_path.'css/admin.css');
	}

	public function hookHeader()
	{	// for 1.4
		return $this->hookDisplayHeader();
	}

/********************************************************************************************
 *	Ajout au header du site pour une conversion au premier visiteur après la CRON
 ********************************************************************************************/

	public function hookDisplayHeader()
	{
		//	Get cron date process
		$date = Configuration::get('NEOTAXESUPDATER_CRONDATE');

		if (Configuration::get('NEOTAXESUPDATER_MANUALCRONMODE', 0) == 0)
		{
			$date = Configuration::get('NEOTAXESUPDATER_CRONDATE');

			//	echo "la date".$date;
			if ($date != '' && $date != '0000-00-00 00:00:00')
			{
				if ($date < date('Y-m-d H:i:s'))
				{
				//	on met à jour la variable de config
					TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_CRONDATE', '');

					$this->runQuietProcess();

				//	clear smarty cache for home featured products
					Context::getContext()->smarty->clearAllCache();
					if (method_exists(Context::getContext()->smarty, 'clearCompiledTemplate'))
						Context::getContext()->smarty->clearCompiledTemplate();

					return '<!-- Ntu refresh --><script type="text/JavaScript">location.reload(true);</script>';
				}
			}
		}
	}

/********************************************************************************************
 *	Fonction qui vérifie les paramètres du serveur et le nombre de produits à traiter
 *	renvoi un message d'alerte si y a un problème
 ********************************************************************************************/
	public function getConfigSlowProblem()
	{
		// On cherche le temps d'exécution serveur
		$max_execution_time = ini_get('max_execution_time');

		// On cherche le nombre de produits concernés
		$nb_prod = TVAUpdater::getNbProducts();

		// Si on est au dessus des 20 produits par secondes, on affiche une alerte!
		if ($nb_prod / $max_execution_time > 10)
		{
			$estimated_time = $nb_prod / 10;
			$estimated_time = (int)$estimated_time;
			return $this->l('Your server is too slow for this option, process will probably be stopped before the end by your server!').'<br/>'.
					$this->l('Ask to Your server administrator to change the max_execution_time value.').'<br/>'.
					sprintf($this->l('The process has to change price for %s products and your server let only %s seconds for that.'),
					$nb_prod, $max_execution_time).'<br/>'.
					sprintf($this->l('Estimated time needed : %s seconds.'), $estimated_time);
		}
		else
			return '';
	}

/********************************************************************************************
 *	Affichage du formulaire sur la page de configuration
 ********************************************************************************************/
	public function displayForm()
	{
		$taxes_list = $this->getAvailableTaxes();

		$this->fields_form[0]['form'] = array(
			'legend' => array(
				'title' => $this->l('Settings'),
				'image' => $this->_path.'logo.gif',
			),
			'description' => $this->l('Remember to save your settings before launching any data process'),
			'input' => array(
				array(
					'type' => 'radio',
					'label' => $this->l('Product prices computing:'),
					'name' => 'taxbehavior',
					'class' => 't',
					'values' => array(
						array(
							'id' => 'taxbehavior_ht',
							'value' => 0,
							'label' => $this->l('Keep price without tax')
						),
						array(
							'id' => 'taxbehavior_ttc',
							'value' => 1,
							'label' => $this->l('Keep price with tax')
						)
					),
					'desc' => $this->l('Define the computing method for product price (products, attributes and specific prices):').'<br />
						<b>'.$this->l('Keep price without tax:').'</b> '.
						$this->l('Will keep price without tax and compute a new price with tax (eg: before 10€ + 10% = 11€, after 10€ + 20% = 12€)').'<br />
						<b>'.$this->l('Keep price with tax:').'</b> '.
						$this->l('Will compute a price without tax and keep price with tax value (eg: before 10€ + 10% = 11€, after 9,166€ + 20% = 11€)'),
	// Not needed anymore for the moment but keep it for next time
	// '<span class="warn" id="slow-server-warning">'.$this->getConfigSlowProblem().'</span>',

				),
				array(
					'type' => 'tax',
					'label' => $this->l('Taxes:'),
					'name' => 'taxBox',
					'values' => $taxes_list,
					'desc' => $this->l('List and usage of taxes.')
				),
				array(
					'id' => 'serverTime',
					'type' => 'text',
					'label' => $this->l('Server date & time :'),
					'name' => 'serverTime',
					'disabled' => true,
					'size' => 15,
					'desc' => $this->l('This is your current server date and time.').'<br />
							<b>'.$this->l('Please take care about this for your processing date and time running.').'</b><br />
							'.$this->l('Date & time format : Year-month-day hour:min:sec')
				),
				array(
					'type' => 'radio',
					'label' => $this->l('Process launch method:'),
					'name' => 'cronmethod',
					'class' => 't',
					'values' => array(
						array(
							'id' => 'cron_simulated',
							'value' => 0,
							'label' => $this->l('I\'m a newbie, do automatically')
						),
						array(
							'id' => 'cron_manual',
							'value' => 1,
							'label' => $this->l('I\'m an expert and use crontab')
						)
					),
					'desc' => $this->l('Define the launch method for processing:').'<br />
						<b>'.$this->l('I\'m a newbie do automatically:').'</b> '.
						$this->l('The first visitor on your shop after the "beginning date" will launch the process automatically and quietly').'<br />
						<b>'.$this->l('I\'m an expert and use crontab:').'</b> '.
						$this->l('The convertion will be done when you want by a cron task that you must configure manually in server crontab')

				),
			),
			'submit'=>array('title' => $this->l('Save'), 'name'=>'saveConfig'),

		);

		$this->fields_form[1]['form'] = array(
			'legend' => array(
				'title' => $this->l('Newbie mode'),
				'image' => $this->_path.'logo.gif',
			),
			'input' => array(
				array(
					'id' => 'cronDateBox',
					'type' => 'date',
					'label' => $this->l('Begining date:'),
					'name' => 'cronDate',
					'values' => '',
					'size' => 15,
					'desc' => $this->l('This date is used in newbie mode to know from which date the conversion can be done')
				),
			),

		);

		$this->fields_form[2]['form'] = array(
			'legend' => array(
				'title' => $this->l('Expert mode'),
				'image' => $this->_path.'logo.gif',
			),
			'input' => array(
				array(
					'type' => 'text',
					'label' => $this->l('Cron running url:'),
					'name' => 'cronurl',
					'class' => 't',
					'size' => '105',
					'desc' => $this->l('This url can be used to lauch conversion by a cron task. Be carefull to see if server time is correct.')
				),
			),
		);

		// get default form data
		$this->fields_value['taxbehavior'] = (int)Configuration::get('NEOTAXESUPDATER_TAXBEHAVIOR');

		$this->fields_value['cronmethod'] = (int)Configuration::get('NEOTAXESUPDATER_MANUALCRONMODE');

		$this->fields_value['serverTime'] = '';

		$this->fields_value['cronDate'] = ($this->neotaxesupdater_getdefaultrules) ? '2014-01-01 00:00:01' : Configuration::get('NEOTAXESUPDATER_CRONDATE');

		$new_taxes_json = Configuration::get('NEOTAXESUPDATER_NEWTAXES');

		if (isset($new_taxes_json) && $new_taxes_json != '')
			$new_taxes = Tools::jsonDecode($new_taxes_json);

		foreach ($taxes_list as $tax_line)
		{
			if ($this->neotaxesupdater_getdefaultrules)
				$this->fields_value['taxBox['.$tax_line['id_tax'].']'] = $this->getDefaultTaxRule2014($tax_line['rate']);
			elseif (isset($new_taxes) && count($new_taxes) > 0)
				$this->fields_value['taxBox['.$tax_line['id_tax'].']'] = isset($new_taxes->{$tax_line['id_tax']}) ? $new_taxes->{$tax_line['id_tax']} : '';
			else
				$this->fields_value['taxBox['.$tax_line['id_tax'].']'] = '';
		}

		// form helper
		$this->fields_value['cronurl'] = (method_exists('Tools', 'getProtocol') ? Tools::getProtocol() : 'http://').
											htmlentities($_SERVER['HTTP_HOST'], ENT_COMPAT, 'UTF-8').
											__PS_BASE_URI__.'modules/'.$this->name.'/'.$this->name.'-cron.php?token='.
											Tools::substr(Tools::encrypt('neotaxesupdater/'.$this->getCronToken()), 0, 10);

		$default_lang = (int)Configuration::get('PS_LANG_DEFAULT');
		$helper = new HelperForm();
		$helper->module = $this;
		$helper->tpl_vars = array('serverDateTime' => time() * 1000);
		$helper->title = $this->displayName;
		$helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;
		$helper->token = TVAUpdater::getAdminTokenLite('AdminModules');
		$helper->toolbar_scroll = true;
		$helper->default_form_language = $default_lang;
		$helper->allow_employee_form_lang = $default_lang;
		$helper->fields_value = $this->fields_value;
		$helper->toolbar_btn = array(
			// @TODO: show only if date is before 2014 useless after
		'prefetch' =>
			array(
				'desc' => $this->l('Configure with 2014 tax'),
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&UseTVA2014config&token='.TVAUpdater::getAdminTokenLite('AdminModules'),
				'imgclass' => 'neo-bt-prefetch',
			),
		'save' =>
			array(
				'desc' => $this->l('Save'),
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&save'.$this->name.
				'&token='.TVAUpdater::getAdminTokenLite('AdminModules'),
				'imgclass' => 'save neo-bt-save',
			),
		'saveandrun' =>
			array(
				'desc' => $this->l('Save & run now'),
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&saveandrun'.$this->name.
				'&token='.TVAUpdater::getAdminTokenLite('AdminModules'),
				'imgclass' => 'save-and-stay neo-bt-saverun', // save for form submit!
			),

		'faq' =>
			array(
				'href' => 'http://www.neovis.fr/faq-neo-taxes-updater',
				'target' => '_blank',
				'desc' => $this->l('FAQ'),
				'imgclass' => 'neo-bt-faq',
			),
		);

		// show only if lofile exists
		if (TVAUpdater::logFileExists())
			$helper->toolbar_btn['viewlogs'] = array(
				'desc' => $this->l('View logs'),
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&viewlog'.$this->name.
				'&token='.TVAUpdater::getAdminTokenLite('AdminModules'),
				'imgclass' => 'neo-bt-logs',
				'target' => '_blank',
			);

		// show only if rollback possible
		if (TVAUpdater::rollbackExists())
			$helper->toolbar_btn['rollback'] = array(
				'desc' => $this->l('Roll Back'),
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&rollback'.$this->name.
				'&token='.TVAUpdater::getAdminTokenLite('AdminModules'),
				'imgclass' => 'neo-bt-rollback',
				'js' => 'if (confirm(\''.str_replace('\'', '\\\'',
					$this->l('Caution, this roll back affects all conversions made since installing this module.').'\n'.
					$this->l('Consult the online FAQ for more informations.').'\n\n'.
					$this->l('Would you like to undo ALL conversions already done by this module?')
				).'\')){return true;}else{event.preventDefault();}'
			);

		$helper->submit_action = 'submit'.$this->name;
	//	$helper->base_folder = dirname(__FILE__).'/views/templates/admin/_configure/helpers/form/';

		// send module object for translation function for previous version of presta
		if ($this->isVer14AndMore() && !$this->isVer15AndMore())
			$helper->setModuleObject($this);

		return $helper->generateForm($this->fields_form);

	}

/********************************************************************************************
 *	pour la page de configuration
 *	Get content permet l'affichage du formulaire de faire le traitement lorsqu'on soumet aussi
 *	on peut également faire une fonction _postprocess si on veut sortir le traitement, à voir
 ********************************************************************************************/
	public function getContent()
	{
		//	on initialise la variable output pour pas d'erreur notice
		$output = '';

		if ($this->isVer15AndMore() && Shop::isFeatureActive())
		{
			$this->adminDisplayInformation(
											$this->l('You have a multi-shop configuration.').'<br/>'.
											$this->l('Please read informations about that in documentation here :').
											' <a href="http://www.neovis.fr/faq-neo-taxes-updater#multishop" target="_blank">http://www.neovis.fr/faq-neo-taxes-updater#multishop</a>'
										);
		}

		// Test directory rights
		if (TVAUpdater::autoCheckLogWritable() !== true)
		{
			$output .= $this->displayError( sprintf(
				$this->l('Errors cannot write log file %s, please check log directory permissions.'),
				TVAUpdater::getTestLogFile()	) );
		}

		// Test directory rights
		if (TVAUpdater::autoCheckRollbackWritable() !== true)
		{
			$output .= $this->displayError( sprintf(
				$this->l('Errors cannot write log file %s, please check backup directory permissions.'),
				TVAUpdater::getTestLogFile()	) );
		}

		//	on charge les paramètres par défaut pour la TVA de 2014
		if (Tools::isSubmit('UseTVA2014config'))
			$this->neotaxesupdater_getdefaultrules = true;

		//	enregistrement des paramètres
		if (Tools::isSubmit('saveConfig') || Tools::isSubmit('saveConfigAndStay'))
		{
			//	recup du type de conversion 0 => touche a rien, 1 => on doit convertir les prix ht!
			TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_TAXBEHAVIOR', (int)Tools::getValue('taxbehavior', 0));

			//	recup du type de conversion 0 => automatique via hook, 1 => crontab
			$cronmethod = (int)Tools::getValue('cronmethod', 0);
			TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_MANUALCRONMODE', $cronmethod);

			$cron_date = Tools::getValue('cronDate', '');
			if ($cronmethod == 0 && Validate::isDate($cron_date))
				TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_CRONDATE', $cron_date);
			else
				TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_CRONDATE', '');

			//	Récup des nouveaux taux à appliquer pour chaque ancien taux
			//	et enregistrement en base
			$tax_box_args = Tools::getValue('taxBox');
			if (isset($tax_box_args) && is_array($tax_box_args))
			{
				// sanitize args
				foreach ($tax_box_args as $k => $v)
				{
					if ($k != (int)$k || $v == '')
						unset($tax_box_args[$k]); // remove strange args
					else
						$tax_box_args[$k] = (float)str_replace(',', '.', $v);
				}
				TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_NEWTAXES', Tools::jsonEncode($tax_box_args));
			}
			//	affichage du message de confirmation que tout c'est bien passé
			$output .= $this->displayConfirmation($this->l('Settings updated'));

		}

		//	Sauvegarde les paramètres et lance direct la cron
		if (Tools::isSubmit('saveConfigAndStay'))
		{
			$output .= $this->runCronProcess();
			return $output;
		}

		//	On veut afficher le fichier log
		if (Tools::isSubmit('viewlogneotaxesupdater'))
		{
			ob_clean(); // Flush HTML header of 1.4 backoffice
			header('Content-Type: text/plain; charset=utf-8');
			readfile(TVAUpdater::getModPath().'log/'.TVAUpdater::$last_log_file_name);
			die();
		}

		//	On veut afficher le fichier log
		if (Tools::isSubmit('rollbackneotaxesupdater'))
			$output .= $this->runRollbackProcess();

		//	affichage du formulaire et des informations
		return $output.$this->displayForm();
	}

/********************************************************************************************
 *	Fonction qui récupere et affiche les taxes disponibles et leur utilisation au sein de la boutique
 ********************************************************************************************/
	private function getAvailableTaxes()
	{
	//	Liste des tva de la boutique
		$taxes = TVAUpdater::getTaxes(Context::getContext()->language->id, null, false, false);

	//	print'<pre>';print_r($taxes);print'</pre>';

	//	on boucle dessus pour alimenter chaque enregistrement de tax avec son utilisation en base
		if ($taxes && count($taxes) > 0)
			foreach ($taxes as &$tax)
			{

			//	récuperation du nombre de produits ayant cette taxe
				$tax['nb_use_product'] = TVAUpdater::getNbProductsByTax($tax['id_tax']);

			//	Récupération du nombre de transporteur utilisant cette taxe
				$tax['nb_use_carrier'] = TVAUpdater::getNbCarriersByTax($tax['id_tax']);

			//	Récupération du nombre de règles de taxe utilisant cette taxe
				$tax['nb_use_taxrule'] = TVAUpdater::getNbTaxesRulesByTax($tax['id_tax']);

			//	Récupération du nombre de commandes utilisant cette taxe
				$tax['nb_use_order'] = TVAUpdater::getNbOrdersByTax($tax['id_tax']);

			}

	//	retour
		return $taxes;
	}

/********************************************************************************************
 *	Fonction qui renvoi les nouveaux taux de TVA pour 2014
 *	details des nouvelles TVA : http://www.service-public.fr/professionnels-entreprises/actualites/00891.html
 *	TVA 19.6 => 20
 *	TVA 7 => 10 Restauration produits alimentaire, transports, travaux rénovation anciens batiments,
 *	TVA 5.5 => 5.5 (aucun changement)
 *	TVA 7 à 5.5 => Cinéma, theatre, concert, cirque
 *	TVA 7 à 2.1 => pour la corse
 *	on renvoi un message si la boutique utilise un taux qu'on ne sait pas renseigner par exemple le 7%!
 ********************************************************************************************/

	public function getDefaultTaxRule2014($tax_rate)
	{
		switch ($tax_rate)
		{
			case 19.6:
				return 20;

			case 7:
				$this->adminDisplayWarning(
					$this->l('You use a specific rate 7% which can be set to 10% or 5.5% or 2.1%, it depends on what you sell.').'<br/>'.
					$this->l('Please fill the good rate manually. ').'<br/>'.
					$this->l('You can get more infos here :').
					' <a href="http://www.neovis.fr/faq-neo-taxes-updater#tauxtva2014" target="_blank">http://www.neovis.fr/faq-neo-taxes-updater#tauxtva2014</a>'
				);
				return '';

			case 8:
				return 10;
		}
		return '';
	}

/********************************************************************************************
 *	Lance le processus de rollback
 ********************************************************************************************/
	public function runRollbackProcess()
	{ // par défaut affiche les messages sinon on les renvoi
		if (!TVAUpdater::rollbackExists())
			$return = $this->displayError($this->l('No Rollback to do!'));
		else if (TVAUpdater::doRollback())
			$return = $this->displayConfirmation($this->l('Rollback successfull!'));
		else
			$return = $this->displayError($this->l('Rollback error!'));

		//	clear smarty cache for home featured products
		Context::getContext()->smarty->clearAllCache();
		if (method_exists(Context::getContext()->smarty, 'clearCompiledTemplate'))
			Context::getContext()->smarty->clearCompiledTemplate();

		return $return;

	}
/********************************************************************************************
 *	Lance le processus de conversion en mode "premier visiteur" apres la date fournie
 ********************************************************************************************/
	public function runQuietProcess()
	{ // par défaut affiche les messages sinon on les renvoi
		// vide le fichier de log
		file_put_contents(TVAUpdater::getModPath().'log/'.TVAUpdater::$last_log_file_name, '');
		$this->processTVAupdate();
	}
/********************************************************************************************
 *	Lance le processus de conversion en mode "cron"
 ********************************************************************************************/
	public function runCronProcess()
	{ // par défaut affiche les messages sinon on les renvoi

		// vide le fichier de log
		file_put_contents(TVAUpdater::getModPath().'log/'.TVAUpdater::$last_log_file_name, '');

		$log = '';

		// if we get one specified id we process only this flux
		// else we process all active flux
		// traitement qui doit renvoyer true si ok, sinon un tableau avec les erreurs
		$return = $this->processTVAupdate();

		if ($return === true)
			$log .= $this->l('Process : done').'<br/>';
		else
		{
			$log .= $this->l('Errors found : ').'<br/>';
			$log .= implode( '<br/>', $return );
			$log .= '<br/>';
		}

		// Si on a lancé la cron en mode connecté au backoffice on propose un lien de retour au module + contenu du log
		// @TODO: vérifier la compatibilité 1.3 / 1.4

		// on affiche le contenu du log
		$retourlog = Tools::file_get_contents(TVAUpdater::getModPath().'log/'.TVAUpdater::$last_log_file_name);
		if ($retourlog == '')
			$log .= '<br/>'.$this->l('Nothing done!');
		else
			$log .= '<br/>'.$this->l('Operation Log :').'<br/>'.nl2br($retourlog);

		if (isset(Context::getContext()->employee))
			$log .= '<br/><br/><a class="button" href="index.php?tab=AdminModules&configure='.$this->name.
					'&token='.Tools::getAdminToken('AdminModules'.(int)Tab::getIdFromClassName('AdminModules').
					(int)Context::getContext()->employee->id).'">'.$this->l('Back').'</a>';

		//	clear smarty cache for home featured products
			Context::getContext()->smarty->clearAllCache();
			if (method_exists(Context::getContext()->smarty, 'clearCompiledTemplate'))
				Context::getContext()->smarty->clearCompiledTemplate();

		return $log;
	}

/********************************************************************************************
 *	Effectue le traitement de changement de tva tel que le paramétrage l'indique
 *	renvoi true si ok sinon renvoi un tableau des erreurs
 ********************************************************************************************/
	public function processTVAupdate()
	{
	//	Liste des tva de la boutique
		$id_lang = (isset(Context::getContext()->language)) ? Context::getContext()->language->id : Configuration::get('PS_LANG_DEFAULT');
		$taxes_list = TVAUpdater::getTaxes($id_lang, null, false, false);

	//	Récup de la config des tva à modifier
		$new_taxes_json = Configuration::get('NEOTAXESUPDATER_NEWTAXES');
		if (isset($new_taxes_json) && $new_taxes_json != '')
		{
			$new_taxes = Tools::jsonDecode($new_taxes_json);
			if (count($new_taxes) > 0)
			{
				foreach ($taxes_list as $tax_line)
				{
						if (isset($new_taxes->{$tax_line['id_tax']}))
						{
							if ($tax_line['rate'] != $new_taxes->{$tax_line['id_tax']})
							{
								if (count(TVAUpdater::$process_errors_found) > 0) // we stop on error!
									return TVAUpdater::$process_errors_found;

								$this->migrateOldTaxToNewTax((int)$tax_line['id_tax'], (float)$new_taxes->{$tax_line['id_tax']});
							}
							else
								TVAUpdater::logMessage(TVAUpdater::MSG_INFO,
									sprintf( $this->l('Nothing to do with this TVA no rate change ID=%s, old rate=%s, new rate=%s'),
										(int)$tax_line['id_tax'], (float)$tax_line['rate'], $new_taxes->{$tax_line['id_tax']} ) );

						//	On success or nothing to do delete config parameter
							unset($new_taxes->{$tax_line['id_tax']});

						}
				}
			}
			else
				TVAUpdater::logMessage(TVAUpdater::MSG_INFO, $this->l('Nothing to do, no TVA to update!'));
		}
		else
			TVAUpdater::logMessage(TVAUpdater::MSG_INFO, $this->l('Nothing to do, no TVA to update!'));

		//	Update config parameters
			TVAUpdater::updateConfigurationValue('NEOTAXESUPDATER_NEWTAXES', Tools::jsonEncode($new_taxes));

		return count(TVAUpdater::$process_errors_found) > 0 ? TVAUpdater::$process_errors_found : true;
	}


/********************************************************************************************
 *	Fonction qui crée un nouveau taux de TVA s'il n'existe pas déjà
 *	on renvoi l'enregistrement TVA correspondant en base
 *	Dans le cas où le taux est à créer on enregistre sa création
 *	pour pouvoir le supprimer en cas de rollback
 ********************************************************************************************/
	public function getAndCreateNewTaxIfNotExists($rate)
	{
		$id_tax = TVAUpdater::getTaxIdByRate($rate, null, false, false);

		//	on doit créer le taux
		if ($id_tax === false)
		{
			$res = TVAUpdater::insertQuery('tax', array(
				'rate' => (float)$rate,
				'active' => 1
			));

			//	table tax lang
			if ($res == 1)
			{
				//	recuperer l'id de l'insertion
				$id_tax = TVAUpdater::insertedID();

				//	enregistre cet ajout pour le rollback
				TVAUpdater::saveForRollback('tax', 'DELETE', 'id_tax='.(int)$id_tax);

				//	log la taxe créée
				TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax record created ID=%s'), (int)$id_tax));

				//	récupération de toutes les langues
				$languages = Language::getLanguages(false);

				foreach ($languages as $language)
				{
					$res = TVAUpdater::insertQuery('tax_lang', array(
						'id_tax' => (int)$id_tax,
						'id_lang' => (int)$language['id_lang'],
						'name' => 'TVA FR '.$rate.'%'
					));

					//	record tax lang created
					if ($res == 1)
					{
						//	enregistre cet ajout pour le rollback
						TVAUpdater::saveForRollback('tax_lang', 'DELETE', 'id_tax='.(int)$id_tax);

						//	log la taxe créée
						TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax lang record created id_tax=%s, id_lang=%s'),
							(int)$id_tax, (int)$language['id_lang']));

					}
					else
					{
						//	log la taxe non créée
						TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Tax record cannot be created id_tax=%s, id_lang=%s'),
							(int)$id_tax, (int)$language['id_lang']));
					}
				}
			}
			else
			{
				//	log la taxe non créée
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Tax record cannot be created rate=%s'), (float)$rate));
			}
		}
		else
		{

			//	on vérifie que le taux est actif et pas effacé, si c'est le cas on l'active et on enlève sa notion de effacé
			$tax = new Tax($id_tax);

			if ($tax->active == 0 || ($this->isVer15AndMore() && $tax->deleted == 1))
			{
				//	on change l'id_tax dans les règles de taxe
				$res = TVAUpdater::updateQuery('tax', array(
					'active' => 1,
					'deleted' => 0,
				), '`id_tax`='.$id_tax);

				if ($res == 1)
				{
					$deleted = ($this->isVer15AndMore()) ? $tax->deleted : 0; // this option does not exists in 1.4 versions

					//	enregistre cet update pour le rollback
					TVAUpdater::saveForRollback('tax', 'UPDATE', 'id_tax='.(int)$id_tax, array('active'=>(int)$tax->active, 'deleted'=>(int)$deleted));
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax record updated and enabled ID=%s'), (int)$id_tax));
				}
				else
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Tax record cannot be updated and enabled ID=%s'), (int)$id_tax));
			}

		}

		//	@TODO: traiter les erreurs de création
		return ($id_tax > 0) ? new Tax($id_tax) : false;
	}

/********************************************************************************************
 *	Migration de l'ancien taux vers le nouveau taux
 *	@TODO: on ne traite pas encore le cas où on aurait plusieurs taxes existantes avec le même taux!
 *	on le traite à moitié en demandant en entrée un idoldtax et pas un old taux
 ********************************************************************************************/
	public function migrateOldTaxToNewTax($old_tax_id, $new_rate)
	{
		//	on vérifie qu'il y a quelquechose à faire
		if (TVAUpdater::getNbTaxesRulesByTax($old_tax_id) == 0 || $new_rate == '')
		{
			TVAUpdater::logMessage(TVAUpdater::MSG_INFO, sprintf($this->l('Nothing to do for this TVA ID='), (int)$old_tax_id));
			return true;
		}

		//	récuperer le taux de l'ancienne TVA
		$old_tax = new Tax($old_tax_id);

		//	récuperer l'id de la nouvelle TVA
		$new_tax = $this->getAndCreateNewTaxIfNotExists($new_rate);

		// on change le libellé du groupe de taxe si c'est possible sinon c'est pas grave!
		// blabla (19.6%) en blabla (20%)

		$taxes_rules = TVAUpdater::getTaxesRulesGroupByTax($old_tax->id);

		$this->updateTaxesRulesName($old_tax, $new_tax, $taxes_rules);

		// recup du type de conversion 0 => touche a rien, 1 => on doit convertir les prix ht!
		if (Configuration::get('NEOTAXESUPDATER_TAXBEHAVIOR', 0) == 1)
			$this->updateProductPriceWithoutTax($old_tax, $new_tax, $taxes_rules);

		//	on change l'id_tax dans les règles de taxe
		$res = TVAUpdater::updateQuery('tax_rule', array(
			'id_tax' => $new_tax->id,
		), '`id_tax`='.$old_tax_id);

		if ($res == 1)
		{
			//	enregistre cet update pour le rollback
			TVAUpdater::saveForRollback('tax_rule', 'UPDATE', 'id_tax='.(int)$new_tax->id, array('id_tax'=>(int)$old_tax_id));

			//	log la règle modifiée
			TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax rule record updated old id_tax=%s, new id_tax=%s'),
				(int)$old_tax_id, (int)$new_tax->id));

			//	archive l'ancienne taxe
			$res = TVAUpdater::updateQuery('tax', array(
				'active' => 0,
				'deleted' => 1,
			), '`id_tax`='.$old_tax_id);

			if ($res == 1)
			{
				$deleted = ( $this->isVer15AndMore() ) ? $old_tax->deleted : 0; // this option does not exists in 1.4 versions

				//	enregistre cet update pour le rollback
				TVAUpdater::saveForRollback('tax', 'UPDATE', 'id_tax='.(int)$old_tax_id, array('active'=>(int)$old_tax->active, 'deleted'=>(int)$deleted));
				//	log la tva modifiée
				TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax record updated and archived ID=%s'), (int)$old_tax_id));

				//	on change aussi la taxe dans une table qui relie produit aux pays avec une taxe
				$res = TVAUpdater::updateQuery('product_country_tax', array(
					'id_tax' => $new_tax->id,
				), '`id_tax`='.$old_tax_id);

				if ($res == 1)
				{
					//	enregistre cet update pour le rollback
					TVAUpdater::saveForRollback('product_country_tax', 'UPDATE', 'id_tax='.(int)$new_tax->id, array('id_tax'=>(int)$old_tax_id));
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax by country for each product updated old id_tax=%s, new id_tax=%s'),
						(int)$old_tax_id, (int)$new_tax->id));
				}
				else
				{
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Tax by country for each product not updated old id_tax=%s, new id_tax=%s'),
						(int)$old_tax_id, (int)$new_tax->id));
				}
			}
			else
			{
				//	log la tva modifiée
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Tax record updated and archived ID=%s'), (int)$old_tax_id));
			}

		}
		else
		{
			//	log la règle non modifiée
			TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Tax record cannot be updated old id_tax=%s, new id_tax=%s'),
				(int)$old_tax_id, (int)$new_tax->id));
		}

		return (bool)$res;
	}

/********************************************************************************************
 *	Renomme le groupe de taxe si jamais il est mensionné son taux dans son libellé
 ********************************************************************************************/
	private function updateTaxesRulesName($old_tax, $new_tax, $taxes_rules)
	{
		// on change le libellé du groupe de taxe si c'est possible sinon c'est pas grave!
		// blabla (19.6%) en blabla (20%)

		foreach ($taxes_rules as $tax_rule)
		{
			if (strpos( $tax_rule['name'], '('.(float)$old_tax->rate.'%)') !== false)
			{
				$newname = str_replace( '('.(float)$old_tax->rate.'%)', '('.(float)$new_tax->rate.'%)', $tax_rule['name'] );
				//	on change le libellé du groupe de taxe
				$res = TVAUpdater::updateQuery('tax_rules_group', array(
					'name' => $newname,
				), '`id_tax_rules_group`='.$tax_rule['id_tax_rules_group']);

				if ($res == 1)
				{
					//	enregistre cet update pour le rollback
					TVAUpdater::saveForRollback('tax_rules_group', 'UPDATE', 'id_tax_rules_group='.$tax_rule['id_tax_rules_group'],
						array('name' => $tax_rule['name']));

					//	log la règle modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Tax rule group record updated old name=%s, new name=%s'),
						$tax_rule['name'], $newname));
				}
			}
		}
	}

/********************************************************************************************
 *	On conserve le prix du produit TTC on change donc le tarif HT
 *	Si on a plusieurs taux de TVA pour le meme produit on fait rien et on le dit!
 ********************************************************************************************/
	private function updateProductPriceWithoutTax($old_tax, $new_tax, $taxes_rules)
	{
		// Dans un premier temps si on a différent taux de tva pour un meme produit
		// on affiche un warning mais on fait rien pour la taxe / produits concernés par cette taxe!
		// Il faudra plus tard traiter ce cas au besoin et en amont

		// On vérifie que pour chaque tax_rule on a le même taux de TVA
		// si oui on continue sinon on arrete tout
		foreach ($taxes_rules as $tax_rule)
		{
			if (!TVAUpdater::isUniqueTaxByTaxeRule($tax_rule['id_tax_rules_group']))
			{
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('This tax rule has many tax rate, cannot process this ID=%s, NAME=%s'),
					$tax_rule['id_tax_rules_group'], $tax_rule['name']));
				return false;
			}
		}

		$price_difference = (1 + $old_tax->rate / 100) / (1 + $new_tax->rate / 100); // used to compute new price

		//	recuperer la liste des produits en détails par taxe
		$products_to_update = TVAUpdater::getProductsByTaxId($old_tax->id);

		//	on boucle sur chaque produit
		if (is_array($products_to_update) && count($products_to_update) > 0)
		{

			// run conversion in SQL and save rollback too
			$res = TVAUpdater::updateProductsPriceByTaxId($old_tax->id, $price_difference);

			if ($res == 1)
			{

				foreach ($products_to_update as $product_to_update)
				{
				//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf($this->l('Product price without tax updated id_product=%s, old price=%s, new price=%s'),
						(int)$product_to_update['id_product'], $product_to_update['price'], $product_to_update['price'] * $price_difference));
				}
			}
			else
			{
				//	log la tva non modifiée et on s'arrête là
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf($this->l('Products prices without tax cannot be updated for this tax ID=%s, rate=%s'),
					$old_tax->id, $old_tax->rate));
				return false;
			}
		}

		//	on change le prix HT du produit dans chaque SHOP pour conserver le meme prix TTC
		$products_shop_to_update = TVAUpdater::getProductsShopByTaxId($old_tax->id);

		//	on boucle sur chaque produit / shop
		if (is_array($products_shop_to_update) && count($products_shop_to_update) > 0)
		{

			// run conversion in SQL and save rollback too
			$res = TVAUpdater::updateProductsShopPriceByTaxId($old_tax->id, $price_difference);

			if ($res == 1)
			{
				foreach ($products_shop_to_update as $product_shop_to_update)
				{
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf(
						$this->l('Product price without tax by shop updated id_product=%s, id_shop=%s, old price=%s, new price=%s'),
						(int)$product_shop_to_update['id_product'], (int)$product_shop_to_update['id_shop'],
						$product_shop_to_update['price'], $product_shop_to_update['price'] * $price_difference));
				}
			}
			else
			{
				//	log la tva non modifiée et on s'arrête là
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf(
					$this->l('Products prices without tax by shop cannot be updated for this tax ID=%s, rate=%s'), $old_tax->id, $old_tax->rate));
				return false;
			}

		}

		//	on change le prix des déclinaisons également puisque le prix final en dépend aussi.
		$products_attributes_to_update = TVAUpdater::getProductsAttributesByTaxId($old_tax->id);

		//	on boucle sur chaque produit / shop
		if (is_array($products_attributes_to_update) && count($products_attributes_to_update) > 0)
		{
			// run conversion in SQL and save rollback too
			$res = TVAUpdater::updateProductsAttributesPriceByTaxId($old_tax->id, $price_difference);

			if ($res == 1)
			{
				foreach ($products_attributes_to_update as $product_attributes_to_update)
				{
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf(
						$this->l('Product attribute price without tax updated id_product_attribute=%s, old price=%s, new price=%s'),
						(int)$product_attributes_to_update['id_product_attribute'], $product_attributes_to_update['price'],
						$product_attributes_to_update['price'] * $price_difference));
				}
			}
			else
			{
				//	log la tva non modifiée et on s'arrête là
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf(
					$this->l('Product attribute price without tax cannot be updated for this tax ID=%s, rate=%s'), $old_tax->id, $old_tax->rate));
				return false;
			}

		}

		//	on change le prix des déclinaisons également puisque le prix final en dépend aussi.
		$products_attributes_to_update = TVAUpdater::getProductsAttributesShopByTaxId($old_tax->id);

		//	on boucle sur chaque produit / shop
		if (is_array($products_attributes_to_update) && count($products_attributes_to_update) > 0)
		{
			// run conversion in SQL and save rollback too
			$res = TVAUpdater::updateProductsAttributesShopPriceByTaxId($old_tax->id, $price_difference);

			if ($res == 1)
			{
				foreach ($products_attributes_to_update as $product_attributes_to_update)
				{
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf(
						$this->l('Product attribute price without tax by shop updated id_product_attribute=%s, id_shop=%s, old price=%s, new price=%s'),
						(int)$product_attributes_to_update['id_product_attribute'], (int)$product_attributes_to_update['id_shop'],
						$product_attributes_to_update['price'], $product_attributes_to_update['price'] * $price_difference));
				}
			}
			else
			{
				//	log la tva non modifiée et on s'arrête là
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf(
					$this->l('Product attribute price without tax by shop cannot be updated for this tax ID=%s, rate=%s'), $old_tax->id, $old_tax->rate));
				return false;
			}

		}

		//	on change les prix spécifiques également puisque le prix final en dépend aussi.
		$products_specifics_prices_to_update = TVAUpdater::getProductSpecificsByTaxId($old_tax->id);

		//	on boucle sur chaque produit / shop
		if (is_array($products_specifics_prices_to_update) && count($products_specifics_prices_to_update) > 0)
		{
			// run conversion in SQL and save rollback too
			$res = TVAUpdater::updateProductSpecificsPriceByTaxId($old_tax->id, $price_difference);

			if ($res == 1)
			{
				foreach ($products_specifics_prices_to_update as $product_specifics_prices_to_update)
				{
					//	log la tva modifiée
					TVAUpdater::logMessage(TVAUpdater::MSG_OK, sprintf(
						$this->l('Product specific price without tax updated id_specific_price=%s, old price=%s, new price=%s'),
						(int)$product_specifics_prices_to_update['id_specific_price'], $product_specifics_prices_to_update['price'],
						$product_specifics_prices_to_update['price'] * $price_difference));
				}
			}
			else
			{
				//	log la tva non modifiée et on s'arrête là
				TVAUpdater::logMessage(TVAUpdater::MSG_ERROR, sprintf(
					$this->l('Product specific price without tax cannot be updated for this tax ID=%s, rate=%s'),
					$old_tax->id, $old_tax->rate));
				return false;
			}

		}

	}

}
